program NEWTONRAPHSON;
{--------------------------------------------------------------------}
{  Alg2'56.pas   Pascal program for implementing Algorithm 2.5-6     }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 2.5 (Newton-Raphson Iteration).                         }
{  Section   2.4, Newton-Raphson and Secant Methods, Page 84         }
{                                                                    }
{  Algorithm 2.6 (Secant Method).                                    }
{  Section   2.4, Newton-Raphson and Secant Methods, Page 85         }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    Max = 150;
    Pmax = 152;
    FunMax = 9;
    GNmax = 230;
    MaxN = 152;

  type
    PVECTOR = array[0..Pmax] of REAL;
    RVECTOR = array[0..GNmax] of REAL;
    LETTERS = string[200];
    Status = (Computing, Done, More, Working);
    DoSome = (Go, Stop);

  var
    Cond, FunType, Inum, Kcount, K, Meth, Sub: INTEGER;
    Delta, Dp, Epsilon, P0, P1, P2, Pone: REAL;
    Ptwo, Pthree, Y1, Y2, RelErr, Rnum: REAL;
    Xmax, Xmin, Ymax, Ymin: REAL;
    DNpts, GNpts, N: INTEGER;
    VP, VQ: PVECTOR;
    Xg, Yg: RVECTOR;
    Ans, Resp: CHAR;
    Stat, State: Status;
    DoMo: DoSome;
    Mess: LETTERS;

  function F (X: REAL): REAL;
  begin
    case FunType of
      1: 
        F := X * X * X - 3 * X + 2;
      2: 
        F := X * X * X - 3 * X - 2;
      3: 
        F := X * X * X - X + 2;
      4: 
        F := X * X * X - X - 3;
      5: 
        F := 4 * X * X * X - 2 * X - 6;
      6: 
        F := 4 * X * X * X - 52 * X * X + 160 * X - 100;
      7: 
        F := X * EXP(-X);
      8: 
        F := 2 * X - 2 - SIN(X);
      9: 
        F := 1600 * (1 - EXP(-X / 5)) - 160 * X;
    end;
  end;

  function F1 (X: REAL): REAL;
  begin
    case FunType of
      1: 
        F1 := 3 * X * X - 3;
      2: 
        F1 := 3 * X * X - 3;
      3: 
        F1 := 3 * X * X - 1;
      4: 
        F1 := 3 * X * X - 1;
      5: 
        F1 := 12 * X * X - 2;
      6: 
        F1 := 12 * X * X - 104 * X + 160;
      7: 
        F1 := (1 - X) * EXP(-X);
      8: 
        F1 := 2 - COS(X);
      9: 
        F1 := 320 * EXP(-X / 5) - 160;
    end;
  end;

  procedure PRINTFUNCTION (FunType: INTEGER);
  begin
    case FunType of
      1: 
        WRITELN('f(x) = x^3 - 3 x + 2');
      2: 
        WRITELN('f(x) = x^3 - 3 x - 2');
      3: 
        WRITELN('f(x) = x^3 - x + 2');
      4: 
        WRITELN('f(x) = x^3 - x - 3');
      5: 
        WRITELN('f(x) = 4 x^3 - 2 x - 6');
      6: 
        WRITELN('f(x) = 4 x^3 - 52 x^2 + 160 x - 100');
      7: 
        WRITELN('f(x) = x exp(-x)');
      8: 
        WRITELN('f(x) = 2 x - 2 - sin(x)');
      9: 
        WRITELN('f(x) = 1600*(1 - exp(-x/5)) - 160 x');
      else
        WRITELN('f(x) = x^2 - 2 x - 1');
    end;
  end;

  procedure NEWRAP ( {FUNCTION F(X: REAL): REAL;}
                  P0, Delta, Epsilon: REAL; Max: INTEGER; var P1, Dp, Y1, Pone, Ptwo: REAL; var Cond, K: INTEGER);
    const
      Small = 1E-20;
    var
      Df, Y0, RelErr: REAL;
  begin
    K := 0;
    Cond := 0;
    VP[0] := P0;    {Store array of approximation}
    Y0 := F(P0);
    VQ[0] := Y0;
    P1 := P0 + 1;
    while (K < Max) and (Cond = 0) do
      begin
        Df := F1(P0);
        if Df = 0 then
          begin
            Cond := 1;
            Dp := P1 - P0;
            P1 := P0;
          end
        else
          begin
            Dp := Y0 / Df;
            P1 := P0 - Dp;
          end;
        Y1 := F(P1);
        RelErr := ABS(Dp);               { /(ABS(P1)+Small); }
        if (RelErr <= Delta) then
          Cond := 2;
        if (ABS(Y1) < Epsilon) then
          Cond := 3;
        if (RelErr <= Delta) and (ABS(Y1) < Epsilon) then
          Cond := 4;
        P0 := P1;
        Y0 := Y1;
        K := K + 1;
        VP[K] := P0;  {Store array of approximation}
        VQ[K] := F(P0);
        if K = 1 then
          Pone := P0;
        if K = 2 then
          Ptwo := P0;
      end;
  end;

  procedure SECANT (P0, P1, Delta, Epsilon: REAL; var P2, Y2, Dp, Ptwo, Pthree: REAL; var Cond, K: INTEGER);
    const
      Small = 1E-20;
    var
      Df, Y0, Y1, RelErr: REAL;
  begin
    K := 0;
    Cond := 0;
    VP[0] := P0;    {Store array of approximation}
    VP[1] := P1;
    Y0 := F(P0);
    Y1 := F(P1);
    VQ[0] := Y0;
    VQ[1] := Y1;
    while (K < Max) and (Cond = 0) do
      begin
        Df := (Y1 - Y0) / (P1 - P0);
        if Df = 0 then
          begin
            Cond := 1;
            Dp := P1 - P0;
            P2 := P1;
          end
        else
          begin
            Dp := Y1 / Df;
            P2 := P1 - Dp;
          end;
        Y2 := F(P2);
        RelErr := ABS(Dp);              { /(ABS(P2)+Small); }
        if (RelErr <= Delta) then
          Cond := 2;
        if (ABS(Y2) < Epsilon) then
          Cond := 3;
        if (RelErr <= Delta) and (ABS(Y2) < Epsilon) then
          Cond := 4;
        P0 := P1;
        P1 := P2;
        Y0 := Y1;
        Y1 := Y2;
        K := K + 1;
        VP[K + 1] := P1;  {Store array of approximation}
        VQ[K + 1] := F(P1);
        if K = 1 then
          Ptwo := P1;
        if K = 2 then
          Pthree := P1;
      end;
    K := K + 1;
  end;

  procedure MESSAGE (var FunType, Meth: INTEGER; var Delta, Epsilon: REAL);
    var
      I, K: INTEGER;
  begin
    CLRSCR;
    for I := 1 to 6 do
      WRITELN;
    WRITELN('                         ROOT FINDING: SLOPE METHODS');
    WRITELN;
    WRITELN;
    WRITELN('     A slope method will be used to find the roots of the equation  f(x) = 0.');
    WRITELN;
    WRITELN('     Either the Newton-Raphson method or the secant method can be used.');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITELN('                   < 1 > The Newton-Raphson method.');
    WRITELN;
    WRITELN('                   < 2 > The secant method.');
    WRITELN;
    WRITELN;
    WRITELN;
    Mess := '                         SELECT your method <1 or 2> ?  ';
    Meth := 1;
    WRITE(Mess);
    READLN(Meth);
    if Meth < 1 then
      Meth := 1;
    if Meth > 2 then
      Meth := 2;
    CLRSCR;
    for I := 1 to 6 do
      WRITELN;
    if Meth = 1 then
      begin
        WRITELN('     Since the  Newton Raphson  method has been chosen,  one initial');
        WRITELN;
        WRITELN;
        WRITELN('starting value  p   must be given. Then the sequence of points  {p }');
        WRITELN('                 0                                                k ');
        WRITELN;
        WRITELN('is generated, where  p  is computed with the Newton-Raphson formula:');
        WRITELN('                      k');
        WRITELN;
        WRITELN;
        WRITELN('        p   =  p    -  f(p   )/f`(p   )      for  k = 1,2,... .');
        WRITELN('         k      k-1       k-1      k-1');
      end
    else
      begin
        WRITELN('     Since the  secant method  has been chosen, two initial starting');
        WRITELN;
        WRITELN;
        WRITELN('values  p  and  p  must be given. Then the sequence of points  {p }');
        WRITELN('         0       1                                               k ');
        WRITELN;
        WRITELN('is generated, where p    is computed with the secant method formula:');
        WRITELN('                     k+1');
        WRITELN;
        WRITELN;
        WRITELN('p    =  p  -  f(p )*[p  - p   ]/[f(p ) - f(p   )]    for  k = 1,2,... .');
        WRITELN(' k+1     k       k    k    k-1      k       k-1');
      end;
    WRITELN;
    WRITELN;
    WRITE('                   Press the <ENTER> key.  ');
    READLN(Resp);
    CLRSCR;
    for I := 1 to 4 do
      WRITELN;
    WRITELN('     Convergence is declared when the difference in consecutive');
    WRITELN;
    WRITELN('     iterates is small,  or the function value is small,  i.e.');
    WRITELN;
    WRITELN;
    WRITELN('          |p  - p   | < Delta    OR    |f(p )| < Epsilon.');
    WRITELN('            N    N-1                       N       ');
    WRITELN;
    WRITELN;
    WRITELN('     Now give values for  Delta and Epsilon.');
    WRITELN;
    WRITELN;
    WRITELN;
    Mess := '     ENTER  the  tolerance   Delta = ';
    Delta := 0.000000001;
    WRITE(Mess);
    READLN(Delta);
    Delta := ABS(Delta);
    if (Delta < 0.000000001) then
      Delta := 0.000000001;
    WRITELN;
    WRITELN;
    Mess := '     ENTER the tolerance   Epsilon = ';
    Epsilon := 0.000000001;
    WRITE(Mess);
    READLN(Epsilon);
    Epsilon := ABS(Epsilon);
    if (Epsilon < 0.000000001) then
      Epsilon := 0.000000001;
    CLRSCR;
    if Meth = 1 then
      WRITELN('     You chose the Newton-Raphson method to find roots of f(x) = 0.')
    else
      WRITELN('     You chose the secant method to find roots of  f(x) = 0.');
    WRITELN;
    WRITELN;
    for K := 1 to FunMax do
      begin
        WRITE('     <', K : 2, ' >   ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    Mess := '     SELECT your function  < 1 - 9 > ?  ';
    FunType := 1;
    WRITE(Mess);
    READLN(FunType);
    if FunType < 1 then
      FunType := 1;
    if FunType > FunMax then
      FunType := FunMax;
  end;

  procedure GETPOINTS (var P0, P1, Delta, Epsilon: REAL; Meth: INTEGER);
    var
      T: REAL;
      Resp: CHAR;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    if Meth = 1 then
      WRITELN('     You chose the Newton-Raphson method to find a zero of:')
    else
      WRITELN('     You chose the secant method to find a zero of:');
    WRITELN;
    WRITE('     ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN;
    case Meth of
      1: 
        begin
          WRITELN('     One initial  starting value  p   must be given.');
          WRITELN('                                   0');
        end;
      2: 
        begin
          WRITELN('     Two starting values  p  and  p   must be given.');
          WRITELN('                           0       1');
        end;
    end;
    WRITELN;
    WRITELN;
    Mess := '     ENTER the initial approx.    p0 = ';
    WRITE(Mess);
    READLN(P0);
    WRITELN;
    P1 := P0 + 1;
    if Meth = 2 then
      begin
        Mess := '     ENTER the initial approx.    p1 = ';
        WRITE(Mess);
        READLN(P1);
        WRITELN;
        if P1 = P0 then
          begin
            if P0 = 0 then
              P1 := 0.001
            else
              P1 := P0 * 1.001 + 0.001 * P0 / ABS(P0);
          end;
      end;
    WRITELN;
  end;                                      {End of PROCEDURE GETPOINT}

  procedure RESULT (P0, Delta, Epsilon, P1, P2, Dp, Y1, Y2: REAL; Pone, Ptwo, Pthree: REAL; Cond, K, Meth: INTEGER);
  begin
    CLRSCR;
    WRITELN;
    case Meth of
      1: 
        WRITELN('The Newton-Raphson method was used to find a zero of');
      2: 
        WRITELN('The secant method was used to find a zero of');
    end;
    WRITELN;
    PRINTFUNCTION(FunType);
    WRITELN;
    if Meth = 1 then
      begin
        WRITELN('starting with  p0 =', P0 : 15 : 7);
        if K > 0 then
          WRITE('then computing p1 =', Pone : 15 : 7);
        if K > 1 then
          WRITE('  and p2 =', Ptwo : 15 : 7);
        if K > 0 then
          WRITELN;
      end
    else
      begin
        WRITELN('starting with  p0 =', P0 : 15 : 7, '  and p1 =', P1 : 15 : 7);
        if K > 0 then
          WRITE('then computing p2 =', Ptwo : 15 : 7);
        if K > 1 then
          WRITE('  and p3 =', Pthree : 15 : 7);
        if K > 0 then
          WRITELN;
      end;
    WRITELN;
    WRITELN('After ', K, ' iterations an approximation for the zero is:');
    WRITELN;
    if Meth = 1 then
      WRITELN('     P  =', P1 : 15 : 7)
    else
      WRITELN('     P  =', P2 : 15 : 7);
    WRITELN;
    WRITELN('    DP  =', ABS(Dp) : 15 : 7, '  is the estimated accuracy for P.');
    WRITELN;
    WRITELN('  Delta =', Delta : 15 : 7);
    WRITELN;
    if Meth = 1 then
      WRITELN('       f(', P1 : 15 : 7, '  )  =', Y1 : 15 : 7)
    else
      WRITELN('       f(', P2 : 15 : 7, '  )  =', Y2 : 15 : 7);
    WRITELN;
    WRITELN('                        Epsilon =', Epsilon : 15 : 7);
    WRITELN;
    if ((Y1 = 0) and (Meth = 1)) or ((Y2 = 0) and (Meth = 2)) then
      begin
        WRITELN('The computed function value is exactly zero!');
      end;
    case Cond of
      0: 
        begin
          WRITELN('Convergence is doubtful because it took ', Max : 2, ' iterations.');
          WRITELN('Hence, the maximum number of iterations was exceeded.');
        end;
      1: 
        begin
          WRITELN('Convergence is doubtful because division by zero was encountered.');
        end;
      2: 
        begin
          WRITELN('The approximation P is within the desired tolerance,');
          WRITELN('because consecutive iterates are closer than  Delta.');
        end;
      3: 
        begin
          WRITELN('The computed value  |f(P)|  is smaller than  Epsilon.');
        end;
      4: 
        begin
          WRITELN('Both the approximation P and the function value f(P) ');
          WRITELN('are within the desired tolerances.');
        end;
    end;
  end;

  procedure PRINTAPPROXS;
    var
      J: INTEGER;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('           k             p                    f(p )');
    WRITELN('                          k                      k ');
    WRITELN('         -------------------------------------------------');
    WRITELN;
    for J := 0 to K do
      begin
        WRITELN('          ', J : 2, '     ', VP[J] : 15 : 7, '     ', VQ[J] : 15 : 7);
        WRITELN;
        if J mod 11 = 9 then
          begin
            WRITE('                  Press  the  <ENTER>  key.  ');
            READLN(Ans);
            WRITELN;
          end;
      end;
    WRITELN;
    WRITE('                  Press  the  <ENTER>  key.  ');
    READLN(Ans);
  end;

begin                                            {Begin Main Program}
  Meth := 1;
  FunType := 1;
  P0 := 1;
  Stat := Working;
  while (Stat = Working) do
    begin
      MESSAGE(FunType, Meth, Delta, Epsilon);
      State := Computing;
      while (State = Computing) do
        begin
          GETPOINTS(P0, P1, Delta, Epsilon, Meth);
          if Meth = 1 then
            NEWRAP(P0, Delta, Epsilon, Max, P1, Dp, Y1, Pone, Ptwo, Cond, K)
          else
            SECANT(P0, P1, Delta, Epsilon, P2, Y2, Dp, Ptwo, Pthree, Cond, K);
          RESULT(P0, Delta, Epsilon, P1, P2, Dp, Y1, Y2, Pone, Ptwo, Pthree, Cond, K, Meth);
          WRITELN;
          WRITE('Do you want to see  all the approximations ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (ANS = 'Y') or (ANS = 'y') then
            PRINTAPPROXS;
          WRITELN;
          case Meth of
            1: 
              WRITE('Want  to try  a  different  starting value ?  <Y/N>  ');
            2: 
              WRITE('Want  to  try  different  starting  values ?  <Y/N>  ');
          end;
          READLN(Resp);
          WRITELN;
          if (Resp <> 'Y') and (Resp <> 'y') then
            State := Done;
          if (Resp = 'Y') or (Resp = 'y') then
            CLRSCR;
        end;
      WRITELN;
      WRITE('Want to try a different function or method ?  <Y/N>  ');
      READLN(Resp);
      if (Resp <> 'Y') and (Resp <> 'y') then
        Stat := Done
    end;
end.                                               {End Main Program}

